module constants
implicit none
double precision, parameter:: hbar = 1.054571817D-34
double precision, parameter:: da_to_kg = 1.6605300000013D-27
double precision, parameter:: bohr_to_m = 5.2917724900001D-11
double precision, parameter:: pi = 3.14159265359D0
double precision, parameter:: j_to_Hz = 1.509190311676D+33
end module constants

module variables
implicit none
double precision:: m1 ! mass of atom 1 - 86.90918053D0 Da for Rb
double precision:: m2 ! mass of atom 2 - 132.90545196D0 Da for Cs
double precision:: a  ! scattering length - 645 bohr for RbCs
double precision, dimension(3,3)::omega_1 ! x, y, z, trap frequencies for atom 1 in kHz (rank 2 tensor)
double precision, dimension(3,3)::omega_2 ! x, y, z, trap frequencies for atom 2 in kHz (rank 2 tensor)
integer:: nx_rel_max ! maximum rel. nx quantum number
integer:: ny_rel_max ! maximum rel. ny quantum number
integer:: nz_rel_max ! maximum rel. nz quantum number
integer:: NX_com_max ! maximum com. nx quantum number
integer:: NY_com_max ! maximum com. ny quantum number
integer:: NZ_com_max ! maximum com. nz quantum number
integer::flag ! flag .eq. (.ne.) 1 selects molecule shifted (unshifted) method

double precision:: mtot
double precision:: mu

double precision, dimension(3,3)::omega_rel  ! rank 2 tesnor of rel motion trap frequencies in kHz
double precision, dimension(3,3)::omega_com  ! rank 2 tensor of com motion trap frequencies in kHz
double precision, dimension(3,3)::omega_couple  !rank 2 tensor for coupling strengths

double precision, dimension(3,3)::beta_rel
double precision, dimension(3,3)::beta_com

integer:: nrel_trap
integer:: ntotal
end module variables

program BOUNDMERGOCOM
use constants
use variables
implicit none
integer:: z0_step, q, i, j, nx_rel, ny_rel, nz_rel, NX_com, NY_com, NZ_com, EE_tot, LWORK, INFO
integer::xflag, yflag, symX, symY, NZ_com_P, nx2_rel, ny2_rel, nz2_rel, l, m, NX_com_P, NY_com_P
integer:: a_int, b, counter, counter_P
double precision:: z0_min, z0_max, E_COM_000, z0, mol_shift, E_com, E_com_P, exp_reltrap_mol, E_rel, exp_relint_trap
double precision:: reltrap, exp_relint_mol_trap, gen_check, integral, rel_term, term_1, term_2, exp_for_JMH
double precision:: rep_arg_numerator, rep_arg_denominator, exp_relint_trap_diff, rep_arg, rep, term_diag, term_off_diag
double precision:: diag_1, diag_2, off_diag_1, off_diag_2, off_diag_3, off_diag_4, off_diag_5
double precision, dimension(:,:,:,:,:,:,:,:), allocatable:: H_array, S_array
double precision, dimension(:,:), allocatable:: overlap_rel_array, overlap_com_array, overlap_NX_com_array, overlap_NY_com_array,&
        &overlap_ny_rel_array, overlap_nx_rel_array, overlap_NZ_otho_array, overlap_nz_rel_array, overlap_NZ_ortho_array,&
        &unit_matrix, H_sym, S_sym
double precision, dimension(:), allocatable:: E_val, WORK
double precision, dimension(1)::WORK_SIZE
integer, dimension(:,:), allocatable::sym_array

! --- PLACE INITIAL VALUES IN TRAP PARAMETER ARRAYS ---

omega_1 = 0.D0
omega_2 = 0.D0

omega_rel = 0.D0
omega_com = 0.D0
omega_couple = 0.D0

beta_rel = 0.D0
beta_com = 0.D0

! --- READ INPUT ---

open(9, file = 'bound-mergo.input')
read(9,*)m1, m2, a,&
        &omega_1(1,1), omega_1(2,2), omega_1(3,3), omega_2(1,1), omega_2(2,2), omega_2(3,3),&
        &z0_min, z0_max, z0_step,&
        &nx_rel_max, ny_rel_max, nz_rel_max, NX_com_max, NY_com_max, NZ_com_max,&
        &flag
close(9)

! --- CALCULATE TOTAL AND REDUCED MASSES ---

mtot = m1 + m2
mu = m1*m2/mtot

write(6,*)''
write(6,*)'--- reduced mass is', mu, 'Da ---'
write(6,*)''

! --- CALCULATE TRAP PARAMETERS --- 

call trap_params
call trap_energies(E_COM_000)

! --- ALLOCATE OVERLAP AND HAMILTONIAN ARRAYS --- 

write(6,*)
if(flag.eq.1)then
        write(6,*)' **** SHIFTED MOLECULE BASIS SET ****'
else
        write(6,*)' **** UNSHIFTED MOLECULE BASIS SET ****'
end if

nrel_trap = (1 + nx_rel_max)*(1 + ny_rel_max)*(1 + nz_rel_max) 

allocate(H_array(-1:(nrel_trap - 1), 0:NX_com_max, 0:NY_com_max, 0:NZ_com_max,&
        &-1:(nrel_trap - 1), 0:NX_com_max, 0:NY_com_max, 0:NZ_com_max))

allocate(S_array(-1:(nrel_trap - 1), 0:NX_com_max, 0:NY_com_max, 0:NZ_com_max,&
        &-1:(nrel_trap - 1), 0:NX_com_max, 0:NY_com_max, 0:NZ_com_max))

allocate(overlap_rel_array(-1:nrel_trap - 1, -1:nrel_trap - 1))

allocate(overlap_com_array(-1:NZ_com_max + 1, -1:NZ_com_max + 1))
allocate(overlap_NZ_ortho_array(-1:NZ_com_max + 1, -1:NZ_com_max + 1))
allocate(overlap_NX_com_array(-1:NX_com_max + 1, -1:NX_com_max + 1))
allocate(overlap_NY_com_array(-1:NY_com_max + 1, -1:NY_com_max + 1))
allocate(overlap_nx_rel_array(-1:nx_rel_max + 1, -1:nx_rel_max + 1))
allocate(overlap_ny_rel_array(-1:ny_rel_max + 1, -1:ny_rel_max + 1))
allocate(overlap_nz_rel_array(-1:nz_rel_max + 1, -1:nz_rel_max + 1))



ntotal = 0
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = -1, nrel_trap - 1
ntotal = ntotal + 1
end do
end do
end do
end do

allocate(sym_array(2,ntotal))

open(22, file = 'basis-functions.txt')
write(6,*)''
!write(6,*)' --- BASIS FUNCTIONS ---'
!write(6,*)''
!write(6,*)' .... nx_rel .... ny_rel .... nz_rel .... NX_com .... NY_com .... NZ_com .... xflag .... yflag ....'
write(6,*)''
j = 0
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = -1, nrel_trap - 1
call relQN(i, nx_rel, ny_rel, nz_rel)
call symmetry(nx_rel, ny_rel, nz_rel, NX_com, NY_com, NZ_com, xflag, yflag)
!write(6,*)nx_rel, ny_rel, nz_rel, NX_com, NY_com, NZ_com, xflag, yflag
write(22,*)nx_rel, ny_rel, nz_rel, NX_com, NY_com, NZ_com, xflag, yflag
j = j + 1
sym_array(1, j) = xflag
sym_array(2, j) = yflag
end do
end do
end do
end do
close(22)

EE_tot = 0
do i = 1, ntotal
if(sym_array(1,i).eq.111.and.sym_array(2,i).eq.111)then
EE_tot = EE_tot + 1
end if
end do

write(6,*)''
write(6,*)'Total of', ntotal,'basis functions'
write(6,*)'Total of',EE_tot,'basis functions with EE symmetry'

allocate(E_val(EE_tot))

allocate(S_sym(EE_tot, EE_tot))
allocate(H_sym(EE_tot, EE_tot))


open(11, file = 'bound-mergo.output')
open(27, file = 'rep.txt')
open(28, file = 'key-matrix-elements.txt')

! --- LOOP OVER VALUES OF Z0 (REMEMBER INPUT IS ASSUMED TO BE Z0/BETA_REL(3,3) ---
! --- FILLING HAMLITONIAN AND OVERLAP ARRAYS, SOLVING GENERALIZED EIGENVALUE PROBLEM ---

z0 = z0_min!*beta_rel(3,3) !remove '*beta_rel(3,3)' stop using reduced units
do q = 1, z0_step

! --- 'Empty' H_array and S_array for new z0 value ---

H_array = 0.D0
S_array = 0.D0
H_sym = 0.D0
S_sym = 0.D0

! --- Construction of overlap array ---

call overlap_rel(overlap_rel_array, z0)

call overlap_com(overlap_com_array,z0)

do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
S_array(-1, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com) = 1.D0
end do
end do
end do

do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = 0, nrel_trap - 1
S_array(i, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com) = 1.D0
end do
end do
end do
end do

do NZ_com_P = 0, NZ_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = 0, nrel_trap - 1
S_array(-1, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com_P) = overlap_rel_array(-1,i)*overlap_com_array(NZ_com, NZ_com_P)
S_array(i, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) = overlap_rel_array(i,-1)*overlap_com_array(NZ_com_P, NZ_com)
end do
end do
end do
end do
end do

! --- Other useful arrays ---

overlap_NX_com_array = 0.D0
overlap_NY_com_array = 0.D0
overlap_nx_rel_array = 0.D0
overlap_ny_rel_array = 0.D0
overlap_nz_rel_array = 0.D0

do i = 0, NX_com_max + 1
overlap_NX_com_array(i,i) = 1.D0
end do

do i = 0, NY_com_max + 1
overlap_NY_com_array(i,i) = 1.D0
end do

call overlap_com(overlap_NZ_ortho_array,0.D0)

do i = 0, nx_rel_max + 1
overlap_nx_rel_array(i,i) = 1.D0
end do

do i = 0, ny_rel_max + 1
overlap_ny_rel_array(i,i) = 1.D0
end do

do i = 0, nz_rel_max + 1
overlap_nz_rel_array(i,i) = 1.D0
end do


! --- T_{com} + V_{com} plus V_couple for mol-mol elements in shifted molecule method ---

mol_shift = 0.5D0*mu**2*omega_couple(3,3)**2*z0**2/(mtot*omega_com(3,3)**2)*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*(bohr_to_m)**2*j_to_Hz/1000.D0

do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max
E_com = (0.5D0 + NX_com)*omega_com(1,1) + (0.5D0 + NY_com)*omega_com(2,2) + (0.5D0 + NZ_com)*omega_com(3,3)
if(flag.eq.1)then
H_array(-1, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com) = &
        &H_array(-1, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com) + E_com - mol_shift
else
H_array(-1, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com) = &
        &H_array(-1, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com) + E_com
end if
end do
end do
end do

do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
E_com = (0.5D0 + NX_com)*omega_com(1,1) + (0.5D0 + NY_com)*omega_com(2,2) + (0.5D0 + NZ_com)*omega_com(3,3)
do i = 0, nrel_trap - 1
H_array(i, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com) = &
        &H_array(i, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com) + E_com
end do
end do
end do
end do

do NZ_com_P = 0, NZ_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
E_com_P = (0.5D0 + NX_com)*omega_com(1,1) + (0.5D0 + NY_com)*omega_com(2,2) + (0.5D0 + NZ_com_P)*omega_com(3,3)
E_com = (0.5D0 + NX_com)*omega_com(1,1) + (0.5D0 + NY_com)*omega_com(2,2) + (0.5D0 + NZ_com)*omega_com(3,3)
do i = 0, nrel_trap - 1
H_array(-1, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com_P) + E_com_P*&
        &S_array(-1, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com_P)
H_array(i, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) = &
        &H_array(i, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) + E_com*&
        &S_array(i, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P)
end do
end do
end do
end do
end do


! --- Implementation of H_{rel} ---

do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max
H_array(-1, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com) = &
        &H_array(-1, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com) - hbar**2/(2.D0*mu*a**2)*&
        &j_to_HZ/(1000.D0*da_to_kg*bohr_to_m**2) + exp_reltrap_mol(z0)
end do
end do
end do

!write(6,*)'molecule binding energy is', -hbar**2/(2.D0*mu*a**2)*j_to_HZ/(1000.D0*da_to_kg*bohr_to_m**2)

do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = 0, nrel_trap - 1
call relQN(i, nx_rel, ny_rel, nz_rel)
E_rel = (0.5D0 + nx_rel)*omega_rel(1,1) + (0.5D0 + ny_rel)*omega_rel(2,2) + (0.5D0 + nz_rel)*omega_rel(3,3)
H_array(i, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com) = &
        &H_array(i, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com) + E_rel 
end do
end do
end do
end do

do j = 0, nrel_trap -1
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = 0, nrel_trap - 1
call relQN(i, nx_rel, ny_rel, nz_rel)
call relQN(j, nx2_rel, ny2_rel, nz2_rel)
exp_relint_trap = 2.D0*pi*hbar**2*a/mu*reltrap(nx_rel, ny_rel, nz_rel, 0.D0, 0.D0, 0.D0, z0)*&
        &reltrap(nx2_rel, ny2_rel, nz2_rel, 0.D0, 0.D0, 0.D0, z0)*&
        &bohr_to_m*(da_to_kg*bohr_to_m**3*1000.D0)**(-1)*j_to_Hz

H_array(i, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com) = &
        &H_array(i, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com) + exp_relint_trap 

end do
end do
end do
end do
end do

term_1 = 2.D0*pi*hbar**2*a/mu*reltrap(1, 0, 0, 0.D0, 0.D0, 0.D0, z0)*&
        &reltrap(1, 0, 0, 0.D0, 0.D0, 0.D0, z0)*&
        &bohr_to_m*(da_to_kg*bohr_to_m**3*1000.D0)**(-1)*j_to_Hz
term_2 = 2.D0*pi*hbar**2*a/mu*reltrap(0, 0, 0, 0.D0, 0.D0, 0.D0, z0)*&
        &reltrap(0, 0, 0, 0.D0, 0.D0, 0.D0, z0)*&
        &bohr_to_m*(da_to_kg*bohr_to_m**3*1000.D0)**(-1)*j_to_Hz
exp_for_JMH = term_1 - term_2 
!write(6,*)'term for JMH', exp_for_JMH

do NZ_com_P = 0, NZ_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = 0, nrel_trap - 1

call relQN(i, nx_rel, ny_rel, nz_rel)

E_rel = (0.5D0 + nx_rel)*omega_rel(1,1) + (0.5D0 + ny_rel)*omega_rel(2,2) + (0.5D0 + nz_rel)*omega_rel(3,3)

exp_relint_mol_trap = -hbar**2/mu*sqrt(2.D0*pi/a)*reltrap(nx_rel, ny_rel, nz_rel, 0.D0, 0.D0, 0.D0, z0)*&
        &da_to_kg**(-1.D0)*bohr_to_m**(-2.D0)*j_to_Hz*0.001D0

H_array(-1, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, i, NX_com, NY_com, NZ_com_P) + (E_rel*overlap_rel_array(-1, i) + &
        &exp_relint_mol_trap)*overlap_com_array(NZ_com, NZ_com_P)

H_array(i, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) = &
        &H_array(i, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) + (E_rel*overlap_rel_array(i, -1) + &
        &exp_relint_mol_trap)*overlap_com_array(NZ_com_P, NZ_com)

end do
end do
end do
end do
end do

! --- Implementation of H_{couple} ---


if(flag.ne.1)then
do NZ_com_P = 0, NZ_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0 , NY_com_max
do NX_com = 0, NX_com_max
H_array(-1, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) - mu*omega_couple(3,3)*&
        &z0*sqrt(0.5D0)*beta_com(3,3)*(sqrt(dble(NZ_com))*overlap_com_array(NZ_com_P, NZ_com - 1) + &
        &sqrt(dble(NZ_com + 1))*overlap_com_array(NZ_com_P, NZ_com + 1))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
end if


do NX_com_P = 0, NX_com_max
do j = 0, nrel_trap -1
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = 0, nrel_trap - 1
call relQN(i, nx_rel, ny_rel, nz_rel)
call relQN(j, nx2_rel, ny2_rel, nz2_rel)
if(ny_rel.eq.ny2_rel.and.nz_rel.eq.nz2_rel)then
H_array(i, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com, NZ_com) = &
        &H_array(i, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com, NZ_com) + mu*omega_couple(1,1)*&
        &0.5D0*beta_rel(1,1)*beta_com(1,1)*(sqrt(dble(NX_com))*overlap_NX_com_array(NX_com_P, NX_com - 1) + &
        &sqrt(dble(NX_com + 1))*overlap_NX_com_array(NX_com_P, NX_com + 1))*&
        &(sqrt(dble(nx_rel))*overlap_nx_rel_array(nx2_rel,nx_rel - 1) + &
        &sqrt(dble(nx_rel + 1))*overlap_nx_rel_array(nx2_rel, nx_rel + 1))&
        &*da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end if
end do
end do
end do
end do
end do
end do


do NY_com_P = 0, NY_com_max
do j = 0, nrel_trap -1
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = 0, nrel_trap - 1
call relQN(i, nx_rel, ny_rel, nz_rel)
call relQN(j, nx2_rel, ny2_rel, nz2_rel)
if(nx_rel.eq.nx2_rel.and.nz_rel.eq.nz2_rel)then
H_array(i, NX_com, NY_com, NZ_com, j, NX_com, NY_com_P, NZ_com) = &
        &H_array(i, NX_com, NY_com, NZ_com, j, NX_com, NY_com_P, NZ_com) + mu*omega_couple(2,2)*&
        &0.5D0*beta_rel(2,2)*beta_com(2,2)*(sqrt(dble(NY_com))*overlap_NY_com_array(NY_com_P, NY_com - 1) + &
        &sqrt(dble(NY_com + 1))*overlap_NY_com_array(NY_com_P, NY_com + 1))*&
        &(sqrt(dble(ny_rel))*overlap_ny_rel_array(ny2_rel,ny_rel - 1) + &
        &sqrt(dble(ny_rel + 1))*overlap_ny_rel_array(ny2_rel, ny_rel + 1))&
        &*da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end if
end do
end do
end do
end do
end do
end do

do NZ_com_P = 0, NZ_com_max
do j = 0, nrel_trap -1
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = 0, nrel_trap - 1
call relQN(i, nx_rel, ny_rel, nz_rel)
call relQN(j, nx2_rel, ny2_rel, nz2_rel)
if(nx_rel.eq.nx2_rel.and.ny_rel.eq.ny2_rel)then
H_array(i, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com_P) = &
        &H_array(i, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com_P) + mu*omega_couple(3,3)*&
        &0.5D0*beta_rel(3,3)*beta_com(3,3)*(sqrt(dble(NZ_com))*overlap_NZ_ortho_array(NZ_com_P, NZ_com - 1) + &
        &sqrt(dble(NZ_com + 1))*overlap_NZ_ortho_array(NZ_com_P, NZ_com + 1))*&
        &(sqrt(dble(nz_rel))*overlap_nz_rel_array(nz2_rel,nz_rel - 1) + &
        &sqrt(dble(nz_rel + 1))*overlap_nz_rel_array(nz2_rel, nz_rel + 1))&
        &*da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end if
end do
end do
end do
end do
end do
end do


do j = 0, nrel_trap -1
call relQN(j, nx_rel, ny_rel, nz_rel)
call relQNinv(nx_rel - 1, ny_rel, nz_rel, l)
call relQNinv(nx_rel + 1, ny_rel, nz_rel, m)
if((nx_rel - 1).eq.-1)then
do NZ_com_P = 0, NZ_com_max
do NX_com_P = 0, NX_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
H_array(-1, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com, NZ_com_P) + mu*omega_couple(1,1)*&
        &0.5D0*beta_com(1,1)*(sqrt(dble(NX_com))*overlap_NX_com_array(NX_com_P, NX_com - 1) + &
        &sqrt(dble(NX_com + 1))*overlap_NX_com_array(NX_com_P, NX_com + 1))*&
        &overlap_com_array(NZ_com, NZ_com_P)*&
        &beta_rel(1,1)*(sqrt(dble(nx_rel))*0.D0 + sqrt(dble(nx_rel + 1))*overlap_rel_array(-1, m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
H_array(j, NX_com, NY_com, NZ_com, -1, NX_com_P, NY_com, NZ_com_P) = &
        &H_array(j, NX_com, NY_com, NZ_com, -1, NX_com_P, NY_com, NZ_com_P) + mu*omega_couple(1,1)*&
        &0.5D0*beta_com(1,1)*(sqrt(dble(NX_com))*overlap_NX_com_array(NX_com_P, NX_com - 1) + &
        &sqrt(dble(NX_com + 1))*overlap_NX_com_array(NX_com_P, NX_com + 1))*&
        &overlap_com_array(NZ_com_P, NZ_com)*&
        &beta_rel(1,1)*(sqrt(dble(nx_rel))*0.D0 + sqrt(dble(nx_rel + 1))*overlap_rel_array(-1, m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
end do
else if((nx_rel).eq.(nx_rel_max))then
rel_term = beta_rel(1,1)*(sqrt(dble(nx_rel))*overlap_rel_array(-1,l) + sqrt(dble(nx_rel + 1))*integral(nx_rel+1,ny_rel,nz_rel,z0))
do NZ_com_P = 0, NZ_com_max
do NX_com_P = 0, NX_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
H_array(-1, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(1,1)*&
        &beta_com(1,1)*(sqrt(dble(NX_com))*overlap_NX_com_array(NX_com_P, NX_com - 1) + &
        &sqrt(dble(NX_com + 1))*overlap_NX_com_array(NX_com_P, NX_com + 1))*&
        &overlap_com_array(NZ_com, NZ_com_P)*rel_term*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
H_array(j, NX_com, NY_com, NZ_com, -1, NX_com_P, NY_com, NZ_com_P) = &
        &H_array(j, NX_com, NY_com, NZ_com, -1, NX_com_P, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(1,1)*&
        &beta_com(1,1)*(sqrt(dble(NX_com))*overlap_NX_com_array(NX_com_P, NX_com - 1) + &
        &sqrt(dble(NX_com + 1))*overlap_NX_com_array(NX_com_P, NX_com + 1))*&
        &overlap_com_array(NZ_com_P, NZ_com)*rel_term*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
end do
else
do NZ_com_P = 0, NZ_com_max
do NX_com_P = 0, NX_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
H_array(-1, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(1,1)*&
        &beta_com(1,1)*(sqrt(dble(NX_com))*overlap_NX_com_array(NX_com_P, NX_com - 1) + &
        &sqrt(dble(NX_com + 1))*overlap_NX_com_array(NX_com_P, NX_com + 1))*&
        &overlap_com_array(NZ_com, NZ_com_P)*&
        &beta_rel(1,1)*(sqrt(dble(nx_rel))*overlap_rel_array(-1,l) + sqrt(dble(nx_rel + 1))*overlap_rel_array(-1,m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
H_array(j, NX_com, NY_com, NZ_com, -1, NX_com_P, NY_com, NZ_com_P) = &
        &H_array(j, NX_com, NY_com, NZ_com, -1, NX_com_P, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(1,1)*&
        &beta_com(1,1)*(sqrt(dble(NX_com))*overlap_NX_com_array(NX_com_P, NX_com - 1) + &
        &sqrt(dble(NX_com + 1))*overlap_NX_com_array(NX_com_P, NX_com + 1))*&
        &overlap_com_array(NZ_com_P, NZ_com)*&
        &beta_rel(1,1)*(sqrt(dble(nx_rel))*overlap_rel_array(-1,l) + sqrt(dble(nx_rel + 1))*overlap_rel_array(-1,m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
end do
end if
end do



do j = 0, nrel_trap -1
call relQN(j, nx_rel, ny_rel, nz_rel)
call relQNinv(nx_rel, ny_rel - 1, nz_rel, l)
call relQNinv(nx_rel, ny_rel + 1, nz_rel, m)
if((ny_rel - 1).eq.-1)then
do NZ_com_P = 0, NZ_com_max
do NY_com_P = 0, NY_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com_P, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com_P, NZ_com_P) + 0.5D0*mu*omega_couple(2,2)*&
        &beta_com(2,2)*(sqrt(dble(NY_com))*overlap_NY_com_array(NY_com_P, NY_com - 1) + &
        &sqrt(dble(NY_com + 1))*overlap_NY_com_array(NY_com_P, NY_com + 1))*&
        &overlap_com_array(NZ_com, NZ_com_P)*&
        &beta_rel(2,2)*(sqrt(dble(ny_rel))*0.D0 + sqrt(dble(ny_rel + 1))*overlap_rel_array(-1, m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com_P, NZ_com_P) = &
        &H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com_P, NZ_com_P) + 0.5D0*mu*omega_couple(2,2)*&
        &beta_com(2,2)*(sqrt(dble(NY_com))*overlap_NY_com_array(NY_com_P, NY_com - 1) + &
        &sqrt(dble(NY_com + 1))*overlap_NY_com_array(NY_com_P, NY_com + 1))*&
        &overlap_com_array(NZ_com_P, NZ_com)*&
        &beta_rel(2,2)*(sqrt(dble(ny_rel))*0.D0 + sqrt(dble(ny_rel + 1))*overlap_rel_array(-1, m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
end do
else if((ny_rel).eq.(ny_rel_max))then
rel_term = beta_rel(2,2)*(sqrt(dble(ny_rel))*overlap_rel_array(-1,l) + sqrt(dble(ny_rel + 1))*integral(nx_rel,ny_rel+1,nz_rel,z0))
do NZ_com_P = 0, NZ_com_max
do NY_com_P = 0, NY_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com_P, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com_P, NZ_com_P) + 0.5D0*mu*omega_couple(2,2)*&
        &beta_com(2,2)*(sqrt(dble(NY_com))*overlap_NY_com_array(NY_com_P, NY_com - 1) + &
        &sqrt(dble(NY_com + 1))*overlap_NY_com_array(NY_com_P, NY_com + 1))*&
        &overlap_com_array(NZ_com, NZ_com_P)*rel_term*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com_P, NZ_com_P) = &
        &H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com_P, NZ_com_P) + 0.5D0*mu*omega_couple(2,2)*&
        &beta_com(2,2)*(sqrt(dble(NY_com))*overlap_NY_com_array(NY_com_P, NY_com - 1) + &
        &sqrt(dble(NY_com + 1))*overlap_NY_com_array(NY_com_P, NY_com + 1))*&
        &overlap_com_array(NZ_com_P, NZ_com)*rel_term*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
end do
else
do NZ_com_P = 0, NZ_com_max
do NY_com_P = 0, NY_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com_P, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com_P, NZ_com_P) + 0.5D0*mu*omega_couple(2,2)*&
        &beta_com(2,2)*(sqrt(dble(NY_com))*overlap_NY_com_array(NY_com_P, NY_com - 1) + &
        &sqrt(dble(NY_com + 1))*overlap_NY_com_array(NY_com_P, NY_com + 1))*&
        &overlap_com_array(NZ_com, NZ_com_P)*&
        &beta_rel(2,2)*(sqrt(dble(ny_rel))*overlap_rel_array(-1,l) + sqrt(dble(ny_rel + 1))*overlap_rel_array(-1,m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com_P, NZ_com_P) = &
        &H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com_P, NZ_com_P) + 0.5D0*mu*omega_couple(2,2)*&
        &beta_com(2,2)*(sqrt(dble(NY_com))*overlap_NY_com_array(NY_com_P, NY_com - 1) + &
        &sqrt(dble(NY_com + 1))*overlap_NY_com_array(NY_com_P, NY_com + 1))*&
        &overlap_com_array(NZ_com_P, NZ_com)*&
        &beta_rel(2,2)*(sqrt(dble(ny_rel))*overlap_rel_array(-1,l) + sqrt(dble(ny_rel + 1))*overlap_rel_array(-1,m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
end do
end if
end do

do j = 0, nrel_trap -1
call relQN(j, nx_rel, ny_rel, nz_rel)
call relQNinv(nx_rel, ny_rel, nz_rel - 1, l)
call relQNinv(nx_rel, ny_rel, nz_rel + 1, m)
if((nz_rel - 1).eq.-1)then
do NZ_com_P = 0, NZ_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(3,3)*&
        &beta_com(3,3)*(sqrt(dble(NZ_com_P))*overlap_com_array(NZ_com, NZ_com_P - 1) + &
        &sqrt(dble(NZ_com_P + 1))*overlap_com_array(NZ_com, NZ_com_P + 1))*&
        &beta_rel(3,3)*(sqrt(dble(nz_rel))*0.D0 + sqrt(dble(nz_rel + 1))*overlap_rel_array(-1, m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) = &
        &H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(3,3)*&
        &beta_com(3,3)*(sqrt(dble(NZ_com))*overlap_com_array(NZ_com_P, NZ_com - 1) + &
        &sqrt(dble(NZ_com + 1))*overlap_com_array(NZ_com_P, NZ_com + 1))*&
        &beta_rel(3,3)*(sqrt(dble(nz_rel))*0.D0 + sqrt(dble(nz_rel + 1))*overlap_rel_array(-1, m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
else if(nz_rel.eq.nz_rel_max)then
rel_term = beta_rel(3,3)*(sqrt(dble(nz_rel))*overlap_rel_array(-1,l) + sqrt(dble(nz_rel + 1))*integral(nx_rel,ny_rel,nz_rel+1,z0))
do NZ_com_P = 0, NZ_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(3,3)*&
        &beta_com(3,3)*(sqrt(dble(NZ_com_P))*overlap_com_array(NZ_com, NZ_com_P - 1) + &
        &sqrt(dble(NZ_com_P + 1))*overlap_com_array(NZ_com, NZ_com_P + 1))*rel_term*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) = &
        &H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(3,3)*&
        &beta_com(3,3)*(sqrt(dble(NZ_com))*overlap_com_array(NZ_com_P, NZ_com - 1) + &
        &sqrt(dble(NZ_com + 1))*overlap_com_array(NZ_com_P, NZ_com + 1))*rel_term*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
else
do NZ_com_P = 0, NZ_com_max
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com_P) = &
        &H_array(-1, NX_com, NY_com, NZ_com, j, NX_com, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(3,3)*&
        &beta_com(3,3)*(sqrt(dble(NZ_com_P))*overlap_com_array(NZ_com, NZ_com_P - 1) + &
        &sqrt(dble(NZ_com_P + 1))*overlap_com_array(NZ_com, NZ_com_P + 1))*&
        &beta_rel(3,3)*(sqrt(dble(nz_rel))*overlap_rel_array(-1,l) + sqrt(dble(nz_rel + 1))*overlap_rel_array(-1,m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) = &
        &H_array(j, NX_com, NY_com, NZ_com, -1, NX_com, NY_com, NZ_com_P) + 0.5D0*mu*omega_couple(3,3)*&
        &beta_com(3,3)*(sqrt(dble(NZ_com))*overlap_com_array(NZ_com_P, NZ_com - 1) + &
        &sqrt(dble(NZ_com + 1))*overlap_com_array(NZ_com_P, NZ_com + 1))*&
        &beta_rel(3,3)*(sqrt(dble(nz_rel))*overlap_rel_array(-1,l) + sqrt(dble(nz_rel + 1))*overlap_rel_array(-1,m))*&
        &da_to_kg*(2.D0*pi*1000.D0)**2*bohr_to_m**2*j_to_Hz/1000.D0
end do
end do
end do
end do
end if
end do

! --- Printing key matrix elements ---

do i = 0, nrel_trap - 1
call relQN(i, nx_rel, ny_rel, nz_rel)
if(nx_rel.eq.0.and.ny_rel.eq.0.and.nz_rel.eq.0)then
off_diag_1 = H_array(-1, 0, 0, 0, i, 0, 0, 1)
off_diag_3 = H_array(-1, 0, 0, 1, i, 0, 0, 1)
diag_1 = H_array(i, 0, 0, 1, i, 0, 0, 1)
end if
if(nx_rel.eq.0.and.ny_rel.eq.0.and.nz_rel.eq.1)then
off_diag_2 = H_array(-1, 0, 0, 0, i, 0, 0, 0)
off_diag_4 = H_array(-1, 0, 0, 1, i, 0, 0, 0)
diag_2 = H_array(i, 0, 0, 0, i, 0, 0, 0)
end if
end do

do i = 0, nrel_trap - 1
call relQN(i, nx_rel, ny_rel, nz_rel)
do j = 0, nrel_trap - 1
call relQN(j, nx2_rel, ny2_rel, nz2_rel)
if(nx_rel.eq.0.and.ny_rel.eq.0.and.nz_rel.eq.0.and.nx2_rel.eq.0.and.ny2_rel.eq.0.and.nz2_rel.eq.1)then
off_diag_5 = H_array(i, 0, 0, 1, j, 0, 0, 0)
end if
end do
end do

write(28,*)z0, H_array(-1, 0, 0, 0, -1, 0, 0, 0), H_array(-1, 0, 0, 0, -1, 0, 0, 1), H_array(-1, 0, 0, 1, -1, 0, 0, 1),&
        &diag_1, diag_2, off_diag_1, off_diag_2, off_diag_3, off_diag_4, off_diag_5



! --- Constructing a hamiltonian matrix of states with EE symmetry ---

b = 0
counter_P = 0
do NZ_com_P = 0, NZ_com_max
do NY_com_P = 0, NY_com_max
do NX_com_P = 0, NX_com_max 
do j = -1, nrel_trap - 1
counter_P = counter_P + 1

if(sym_array(1,counter_P).eq.111.and.sym_array(2,counter_P).eq.111)then
b = b + 1
end if

a_int = 0
counter = 0
do NZ_com = 0, NZ_com_max
do NY_com = 0, NY_com_max
do NX_com = 0, NX_com_max 
do i = -1, nrel_trap - 1
counter = counter + 1

if(sym_array(1,counter).eq.111.and.sym_array(2,counter).eq.111)then
a_int = a_int + 1
end if

if(sym_array(1,counter_P).eq.111.and.sym_array(2,counter_P).eq.111.and.&
        &sym_array(1,counter).eq.111.and.sym_array(2,counter).eq.111)then
S_sym(a_int, b) = S_array(i, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com_P, NZ_com_P)
H_sym(a_int, b) = H_array(i, NX_com, NY_com, NZ_com, j, NX_com_P, NY_com_P, NZ_com_P)
end if

end do
end do
end do
end do
end do
end do
end do
end do

! --- Calls to MATPRN stored here ---

!call MATPRN(6, overlap_rel_array, nrel_trap + 1, nrel_trap + 1, nrel_trap + 1, 1, overlap_rel_array, &
!        &'overlap_rel_array', overlap_rel_array)

!call MATPRN(6, overlap_com_array, NZ_com_max + 2, NZ_com_max + 2, NZ_com_max + 2, 1, overlap_com_array, &
!        &'overlap_com_array', overlap_com_array)

!call MATPRN(6, S_array, ntotal, ntotal, ntotal, 1, S_array, 'S_array', S_array)

!call MATPRN(6, H_array, ntotal, ntotal, ntotal, 1, H_array, 'H_array', H_array)

!call MATPRN(6, H_sym, EE_tot, EE_tot, EE_tot, 1, H_sym, 'H_array', H_sym)


! --- Diagonalization and eigenvalues output ---

!allocate(unit_matrix(ntotal,ntotal))
!unit_matrix=0.D0
!do j = 1, ntotal
!unit_matrix(j,j) = 1.D0
!end do
allocate(unit_matrix(EE_tot,EE_tot))
unit_matrix=0.D0
do j = 1, EE_tot
unit_matrix(j,j) = 1.D0
end do

!LWORK = -1
!call dsygv(1, 'N', 'U', ntotal, H_array, ntotal, S_array, ntotal, E_val, WORK_SIZE, LWORK, INFO)
!LWORK = nint(WORK_SIZE(1))
!allocate(WORK(LWORK))
LWORK = -1
call dsygv(1, 'N', 'U', EE_tot, H_sym, EE_tot, S_sym, EE_tot, E_val, WORK_SIZE, LWORK, INFO)
LWORK = nint(WORK_SIZE(1))
allocate(WORK(LWORK))

!call dsygv(1, 'N', 'U', ntotal, H_array, ntotal, S_array, ntotal, E_val, WORK, LWORK, INFO)
!deallocate(WORK)
!deallocate(unit_matrix)
!write(6,*)'...',dble(z0), dble(z0)/beta_rel(3,3),'...', (E_val(1:10) - E_com_000),'...'
!write(11,*)dble(z0), (E_val - E_com_000)
call dsygv(1, 'N', 'U', EE_tot, H_sym, EE_tot, S_sym, EE_tot, E_val, WORK, LWORK, INFO)
deallocate(WORK)
deallocate(unit_matrix)
write(6,*)'...',dble(z0), '...', (E_val(1:6)),'...'
write(11,*)dble(z0), (E_val)

! --- Details pertaining to rel-com vs sep-atom representations ---

rep_arg_numerator = mu*da_to_kg*omega_couple(3,3)*(1000.0*2.D0*pi)**2*&
        &beta_rel(3,3)*beta_com(3,3)*bohr_to_m**2*j_to_Hz
exp_relint_trap_diff = 2.D0*pi*hbar**2*a/mu*reltrap(0, 0, 0, 0.D0, 0.D0, 0.D0, z0)*&
        &reltrap(0, 0, 0, 0.D0, 0.D0, 0.D0, z0)*&
        &bohr_to_m*(da_to_kg*bohr_to_m**3*1000.D0)**(-1)*j_to_Hz*&
        & - 2.D0*pi*hbar**2*a/mu*reltrap(0, 0, 1, 0.D0, 0.D0, 0.D0, z0)*&
        &reltrap(0, 0, 1, 0.D0, 0.D0, 0.D0, z0)*&
        &bohr_to_m*(da_to_kg*bohr_to_m**3*1000.D0)**(-1)*j_to_Hz
rep_arg_denominator = omega_com(3,3)*1000.0 - omega_rel(3,3)*1000.0 + exp_relint_trap_diff
rep_arg = rep_arg_numerator/rep_arg_denominator
rep = 0.5D0*atan(rep_arg)

exp_relint_trap = 2.D0*pi*hbar**2*a/mu*reltrap(0, 0, 0, 0.D0, 0.D0, 0.D0, z0)*&
        &reltrap(0, 0, 0, 0.D0, 0.D0, 0.D0, z0)*&
        &bohr_to_m*(da_to_kg*bohr_to_m**3*1000.D0)**(-1)*j_to_Hz/1000.0

rep_arg_numerator = -(1.054571817D-34)**2/(52.54747505*1.660530E-27)*&
        &sqrt(2.E0*exp(-0.5E0*(z0/beta_rel(3,3))**2)&
        &/(pi**0.5D0*554.D0*beta_rel(3,3)**3*(5.29177249D-11)**4))*1.509190179D33/1000.D0
rep_arg = (rep_arg_numerator - exp_relint_trap*overlap_rel_array(-1, 0))/(1.D0 - overlap_rel_array(-1,0)**2)

!write(27,*)z0, rep_arg_numerator, exp_relint_trap_diff, rep_arg_denominator, rep_arg, rep
!write(6,*)z0, rep_arg_numerator, exp_relint_trap_diff, rep_arg_denominator, rep_arg, rep
write(6,*)'...', abs(rep_arg/2.E0), '...'

! --- Calculate next value of z0 and loop back ---


!z0 = z0 + (z0_max*beta_rel(3,3) - z0_min*beta_rel(3,3))/dble(z0_step) ! no longer working with reduced length in input file
z0 = z0 + (z0_max - z0_min)/dble(z0_step)
end do

close(11)
close(27)
close(28)

deallocate(H_array)
deallocate(S_array)
deallocate(overlap_rel_array)
deallocate(overlap_com_array)
deallocate(overlap_nx_rel_array)
deallocate(overlap_ny_rel_array)
deallocate(overlap_nz_rel_array)
deallocate(overlap_NX_com_array)
deallocate(overlap_NY_com_array)
deallocate(overlap_NZ_ortho_array)
deallocate(sym_array)
deallocate(S_sym)
deallocate(H_sym)
end program

subroutine trap_params
use constants
use variables
implicit none
integer:: i

do i = 1, 3
omega_rel(i,i) = dsqrt((m2*omega_1(i,i)**2 + m1*omega_2(i,i)**2)/mtot)
omega_com(i,i) = dsqrt((m1*omega_1(i,i)**2 + m2*omega_2(i,i)**2)/mtot)
omega_couple(i,i) = omega_2(i,i)**2 - omega_1(i,i)**2
end do

do i = 1, 3
beta_rel(i,i) = dsqrt(hbar/(mu*omega_rel(i,i)))*dsqrt(1.D0/(da_to_kg*2.D0*pi*1.D3))*(bohr_to_m)**(-1.D0)
beta_com(i,i) = dsqrt(hbar/(mtot*omega_com(i,i)))*dsqrt(1.D0/(da_to_kg*2.D0*pi*1.D3))*(bohr_to_m)**(-1.D0)
end do

write(6,*)'          --- PRINTING OUT TRAP PARAMETERS FOR THIS CALCULATION --- '
write(6,*)''
write(6,*)'          --------------   x    ---------------   y ---------------    z    ------------'
write(6,*)''
write(6,*)'omega_1      ', omega_1(1, 1), omega_1(2,2), omega_1(3,3)
write(6,*)'omega_2      ', omega_2(1, 1), omega_2(2,2), omega_2(3,3)
write(6,*)''
write(6,*)'omega_rel    ', omega_rel(1, 1), omega_rel(2,2), omega_rel(3,3)
write(6,*)'beta_rel     ', beta_rel(1, 1), beta_rel(2,2), beta_rel(3,3)
write(6,*)''
write(6,*)'omega_com    ', omega_com(1, 1), omega_com(2,2), omega_com(3,3)
write(6,*)'beta_com     ', beta_com(1, 1), beta_com(2,2), beta_com(3,3)
write(6,*)''
write(6,*)'omega_coup   ', omega_couple(1, 1), omega_couple(2,2), omega_couple(3,3)

return
end subroutine trap_params

subroutine trap_energies(E_COM_000)
use variables
use constants
implicit none
integer::nx1, ny1, nz1, nx2, ny2, nz2
double precision:: E_COM, E
double precision, intent(out)::E_COM_000

E_COM_000 = omega_com(1,1)*(0.D0 + 0.5D0) + omega_com(2,2)*(0.D0 + 0.5D0) + omega_com(3,3)*(0.D0 + 0.5D0)

open(13, file = 'com_energies.txt')
do nx1 = 0, nx_com_max
do ny1 = 0, ny_com_max
do nz1 = 0, nz_com_max
E_COM = omega_com(1,1)*(dble(nx1) + 0.5D0) + omega_com(2,2)*(dble(ny1) + 0.5D0) + omega_com(3,3)*(dble(nz1) + 0.5D0)
write(13,*) nx1, ny1, nz1, E_COM, E_COM - E_COM_000
end do
end do
end do
close(13)

open(14, file = 'sep_trap_energies.txt')
do nx1 = 0, 5
do ny1 = 0, 5
do nz1 = 0, 5
do nx2 = 0, 5
do ny2 = 0, 5
do nz2 = 0, 5
E = (dble(nx1) + 0.5D0)*omega_1(1,1) + (dble(ny1) + 0.5D0)*omega_1(2,2) + (dble(nz1) + 0.5D0)*omega_1(3,3)&
        & + (dble(nx2) + 0.5D0)*omega_2(1,1) + (dble(ny2) + 0.5D0)*omega_2(2,2) + (dble(nz2) + 0.5D0)*omega_2(3,3)
if((E - E_COM_000).lt.600.D0)then
        write(14,*) nx1, ny1, nz1, nx2, ny2, nz2, E, E - E_COM_000
end if
end do
end do
end do
end do
end do
end do
close(14)

return
end subroutine

subroutine relQN(i, nx, ny, nz)
use variables
use constants
implicit none
integer:: nxdim, nydim, nzdim, k
integer, intent(in):: i
integer, intent(out):: nx, ny, nz

if (i.eq.-1)then
nx = 999
ny = 999
nz = 999
else
nxdim = 1 + nx_rel_max
nydim = 1 + ny_rel_max
nzdim = 1 + nz_rel_max

nz = i/(nxdim*nydim)
k = i - nxdim*nydim*nz
ny = k/nxdim
nx = k - nxdim*ny
end if

return
end subroutine

subroutine relQNinv(nx, ny, nz, i)
use variables
use constants
implicit none
integer:: nxdim, nydim, nzdim, k
integer, intent(out):: i
integer, intent(in):: nx, ny, nz

if (nx.eq.999.and.ny.eq.999.and.nz.eq.999)then
i = -1
else
nxdim = 1 + nx_rel_max
nydim = 1 + ny_rel_max
nzdim = 1 + nz_rel_max

i = nxdim*nydim*nz + nxdim*ny + nx

end if

return
end subroutine

subroutine symmetry(nx_rel, ny_rel, nz_rel, NX_com, NY_com, NZ_com, xflag, yflag)
implicit none
integer:: check
integer, intent(in)::nx_rel, ny_rel, nz_rel, NX_com, NY_com, NZ_com
integer, intent(out)::xflag, yflag

if(nx_rel.eq.999)then
check = (-1)**NX_com
else
check = (-1)**(nx_rel + NX_com)
end if

if(check.eq.-1)then
xflag = 111
else if(check.eq.1)then
xflag = 100
end if

if(ny_rel.eq.999)then
check = (-1)**NY_com
else
check = (-1)**(ny_rel + NY_com)
end if

if(check.eq.-1)then
yflag = 111
else if(check.eq.1)then
yflag = 100
end if

return
end subroutine


subroutine overlap_rel(overlap, z0)
use variables
use constants
implicit none
double precision, dimension(-1:(nrel_trap - 1), -1:(nrel_trap - 1)), intent(out)::overlap
double precision, intent(in)::z0
integer:: i, j, nx, ny, nz
double precision:: integral

overlap = 0.D0

do i = -1, nrel_trap - 1
overlap(i,i) = 1.D0
end do

do j = 0, nrel_trap - 1
call relQN(j, nx, ny, nz)
overlap(-1, j) = integral(nx, ny, nz, z0)
overlap(j, -1) = overlap(-1, j)
end do

return
end subroutine

function integral(nx, ny, nz, z0)
use constants
use variables
implicit none
integer:: IFAIL, r_loop, t_loop, p_loop, nx, ny, nz
integer, parameter::ITYPE = 1, NPTS = 48
double precision:: B, integral, r, t, p,  integrand, integrand_weighted, z0
double precision:: reltrap, mol
double precision:: WEIGHTS_r(NPTS), ABSCIS_r(NPTS), WEIGHTS_t(NPTS), ABSCIS_t(NPTS)

IFAIL = 0

B = 1.D0/a

call d01bax(0.D0, B, ITYPE, NPTS, WEIGHTS_r, ABSCIS_r, IFAIL) !Laguerre quadrature
call d01baz(1.D0, -1.D0, ITYPE, NPTS, WEIGHTS_t, ABSCIS_t, IFAIL) !Legendre quadrature

integral = 0.D0

do p_loop = 1, int(NPTS)
p = 2.D0*pi*(dble(p_loop) - 0.5D0)/dble(NPTS)
        
do t_loop = 1, int(NPTS)
t = ABSCIS_t(t_loop)
                
do r_loop = 1, int(NPTS)
r = ABSCIS_r(r_loop)
                        
integrand = (-1.D0)*mol(r)*reltrap(nx, ny, nz, r, t, p, z0)*r**2
integrand_weighted = integrand*WEIGHTS_r(r_loop)*WEIGHTS_t(t_loop)*(2.D0*pi/dble(NPTS))
integral = integral + integrand_weighted
                
end do
end do
end do

end function integral

function mol(r)
use constants
use variables
implicit none
double precision:: r, mol
mol = 1.D0/dsqrt(2.D0*pi*a)*(1.D0/r)*dexp(-r/a)
end function

function reltrap(nx, ny, nz, r, t, p, z0)
use constants
use variables
implicit none
integer:: nx, ny, nz
double precision:: r, t, p, x, y, z, psi_x, psi_y, psi_z, z0, reltrap, fact
double precision, dimension(:), allocatable::herm_array_x, herm_array_y, herm_array_z

x = r*sqrt(1.D0 - t**2)*cos(p)
y = r*sqrt(1.D0 - t**2)*sin(p)
z = r*t

allocate(herm_array_x(nx + 1))
allocate(herm_array_y(ny + 1))
allocate(herm_array_z(nz + 1))

call herm(herm_array_x, nx + 1, (x/beta_rel(1,1)))
call herm(herm_array_y, ny + 1, (y/beta_rel(2,2)))
call herm(herm_array_z, nz + 1, ((z - z0)/beta_rel(3,3)))

psi_x = (2.D0**nx*fact(nx)*beta_rel(1,1))**(-0.5D0)*pi**(-0.25D0)*herm_array_x(nx + 1)*exp(-0.5D0*(x/beta_rel(1,1))**2)
psi_y = (2.D0**ny*fact(ny)*beta_rel(2,2))**(-0.5D0)*pi**(-0.25D0)*herm_array_y(ny + 1)*exp(-0.5D0*(y/beta_rel(2,2))**2)
psi_z = (2.D0**nz*fact(nz)*beta_rel(3,3))**(-0.5D0)*pi**(-0.25D0)*herm_array_z(nz + 1)*exp(-0.5D0*((z - z0)/beta_rel(3,3))**2)

reltrap = psi_x*psi_y*psi_z

end function

function fact(n)
implicit none
integer::i, n
double precision:: fact
fact = 1.D0
do i = 2, n
   fact = fact*dble(i)
end do
return
end function fact

subroutine overlap_com(overlap, z0)
use variables
use constants
implicit none
double precision, intent(in)::z0
double precision, dimension(-1:(NZ_com_max + 1), -1:(NZ_com_max + 1)), intent(out)::overlap
double precision, dimension(0:(NZ_com_max + 1), 0:(NZ_com_max + 1)):: lag_matrix
integer:: m, n, alpha, k
double precision:: rho, x, prefactor, gradry

overlap = 0.D0
lag_matrix = 0.D0

if (flag.eq.1)then
rho = mu*omega_couple(3,3)*z0/(mtot*omega_com(3,3)**2*beta_com(3,3))
else
rho = 0.D0
end if

lag_matrix(0, :) = 1.D0

x = 0.5D0*rho**2

do n = 1, NZ_com_max + 1
lag_matrix(1,n) = 1.D0 + dble(n - 1) - x
end do

do m = 2, NZ_com_max + 1
do n = m, NZ_com_max + 1
alpha = n - m
k = m - 1
lag_matrix(m, n) = dble((k + 1.D0)**(-1))*((2.D0*dble(k) + 1.D0 + dble(alpha) - x)*lag_matrix(k,n -1) - &
        &dble(k + alpha)*lag_matrix(k - 1, n - 2))
end do
end do

do m = 0, NZ_com_max + 1
do n = m, NZ_com_max + 1
prefactor = sqrt(gamma(dble(m+1))/(2.D0**(n - m)*gamma(dble(n+1))))*rho**(n - m)*exp(-0.25D0*rho**2)
gradry = prefactor*lag_matrix(m, n)
overlap(m, n) = gradry
overlap(n, m) = (-1.D0)**(n - m)*gradry
end do
end do


return
end subroutine

function exp_reltrap_mol(z0)
use constants
use variables
implicit none
integer:: IFAIL, r_loop, t_loop, p_loop
integer, parameter::ITYPE = 1, NPTS = 48
double precision::B, integral, r, t, p, z0, integrand, integrand_weighted
double precision:: WEIGHTS_r(NPTS), ABSCIS_r(NPTS), WEIGHTS_t(NPTS), ABSCIS_t(NPTS), exp_reltrap_mol, x, y, z, mol, v_harm

IFAIL = 0

B = 2.D0/a

call d01bax(0.D0, B, ITYPE, NPTS, WEIGHTS_r, ABSCIS_r, IFAIL) !Laguerre quadrature
call d01baz(1.D0, -1.D0, ITYPE, NPTS, WEIGHTS_t, ABSCIS_t, IFAIL) !Legendre quadrature

integral = 0.D0
do p_loop = 1, int(NPTS)
p = 2.D0*pi*(dble(p_loop) - 0.5D0)/dble(NPTS)
        
do t_loop = 1, int(NPTS)
t = ABSCIS_t(t_loop)
                
do r_loop = 1, int(NPTS)
r = ABSCIS_r(r_loop)
                        
x = r*sqrt(1.D0 - t**2)*cos(p)
y = r*sqrt(1.D0 - t**2)*sin(p)
z = r*t
                        
v_harm = (0.5D0*mu*omega_rel(1,1)**2*x**2 + 0.5D0*mu*omega_rel(2,2)**2*y**2 + &
             &0.5D0*mu*omega_rel(3,3)**2*(z-z0)**2)
                        
integrand = (-1.D0)*mol(r)*mol(r)*v_harm*r**2
integrand_weighted = integrand*WEIGHTS_r(r_loop)*WEIGHTS_t(t_loop)*(2.D0*pi/dble(NPTS))
integral = integral + integrand_weighted
                
end do
end do
end do

exp_reltrap_mol = integral*da_to_kg*(1000.D0*2.D0*pi)**2*bohr_to_m**2*j_to_Hz*0.001D0
end function

